//////////////////////////////////////////////////////////////////////////////////
//No part of this file can be copied or released without the consent of 
//Avalanche Technology
//										
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
//										
//	Avalanche Technology Inc., Proprietary and Confidential	   *
//										
// 	Release:  2.5    Date 12/11/2024 	
//										
////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////
//  PART DESCRIPTION:
//
//  Technology: 22nm pMTJ STT-MRAM
//  Part:       AS3064204/AS3128208 
//
//  Description: 64/128 Megabit DUAL QSPI PERSISTENT SRAM MEMORY
//  Datasheet Revision : Rev C
//
////////////////////////////////////////////////////////////////////////////////////
//  FILE CONTENTS : MODEL OF SPSRAM 
//
////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////
//  DENSITY SELECTION                                                             //
//                                                                                //
//  Uncomment one of the following lines to select the appropriate density. If    //
//  no line is selected, the model will default to 128Mb (64Mb per lane)          //
//                                                                                //
////////////////////////////////////////////////////////////////////////////////////
//`define DENSITY_64M   // 64Mb   (Single QSPI)
//`define DENSITY_128M  // 128Mb  (Dual QSPI - 64Mb per lane)

////////////////////////////////////////////////////////////////////////////////////
// MODULE DECLARATION                                                             //
////////////////////////////////////////////////////////////////////////////////////

`timescale 100ps / 10ps

module spsram_as3xxx20x
  (
  inout [7:0] IO,
  input CLK,
  output INTn,
  input CSn,
  input [2:0] HBP,
  input HTBSEL,
  input RESETn
  );

  spsram_64m I0(
    .IO3     (IO[3]),
    .WPn_IO2 (IO[2]),
    .SO_IO1  (IO[1]),
    .SI_IO0  (IO[0]),
    .CSn     (CSn),
    .CLK     (CLK),
    .INTn    (INTn),
    .HBP     (HBP),
    .HTBSEL  (HTBSEL),
    .RESETn  (RESETn)
    );

`ifndef DENSITY_64M    
  spsram_64m I1(
    .IO3     (IO[7]),
    .WPn_IO2 (IO[6]),
    .SO_IO1  (IO[5]),
    .SI_IO0  (IO[4]),
    .CSn     (CSn),
    .CLK     (CLK),
    .INTn    (INTn),
    .HBP     (HBP),
    .HTBSEL  (HTBSEL),
    .RESETn  (RESETn)
    );
`endif

endmodule

module spsram_64m
  ( 
  IO3, 
  WPn_IO2, 
  SO_IO1,
  SI_IO0,
  CSn, 
  CLK,
  INTn,
  HBP,
  HTBSEL,
  RESETn
  );
  
  inout IO3; 
  inout SI_IO0; 
  inout SO_IO1;
  inout WPn_IO2;
  input CSn;
  input CLK;
  input [2:0] HBP;
  input HTBSEL;
  input RESETn;
  output INTn;

  // Instruction type
  parameter NOOP = 8'h00;
  parameter WREN = 8'h06;
  parameter WRDI = 8'h04;
  parameter RDSR = 8'h05;
  parameter RDFSR = 8'h70;
  parameter RDID = 8'h9F;
  parameter RDAR = 8'h65;
  parameter WRSR = 8'h01;
  parameter WRAR = 8'h71;
  parameter READ = 8'h13;
  parameter READ1 = 8'h03;
  parameter RDFT = 8'h0C; 
  parameter RDFT1 = 8'h0B; 
  parameter WRTE = 8'h02;
  parameter WRFT = 8'hDA;
  parameter SPIE = 8'hFF;
  parameter QPIE = 8'h38;
  parameter DRFR = 8'h0D;
  parameter DRFW = 8'hDE;
  parameter RDQO = 8'h6C; 
  parameter RDQO1 = 8'h6B; 
  parameter RDQI = 8'hEB;
  parameter WQIO = 8'hD2;
  parameter DRQI = 8'hED;
  parameter DWQO = 8'hD1;

  //SPI DPI and QPI modes
  parameter SPIEN   = 8'hFF;
  parameter QPIEN   = 8'h38;

  ////////////////////////////////////////////////////////////
  // Timing Violation checks
  ///////////////////////////////////////////////////////////
  //$setup, $hold, $width violations
  reg tCSS_notifier;
  reg tCSH_notifier;
  reg tCH_notifier;
  reg tCL_notifier;
  reg tCS_notifier;
  reg tSU_notifier;
  reg tHD_notifier;
  reg tOH_notifier;
  reg tP50_notifier;

  specify 
    specparam tPOR_time  = 1; 
    specparam tCSn_del   = 1; 
    specparam tRESET     = 4500000; 
    specparam tCSS       = 50; 
    specparam tCSH       = 40; 
    specparam tCS        = 4900; 
    specparam tCO        = 70;
    specparam tOH        = 10;
`ifdef DDR_TIMING
    specparam tCH        = 90; //0.45 of freq 50MHz
    specparam tCL        = 90; 
    specparam tSU        = 20;
    specparam tHD        = 30;
`else
    specparam tCH        = 45; //0.45 of freq 100MHz
    specparam tCL        = 45; 
    specparam tSU        = 20;
    specparam tHD        = 30;
`endif

    $setup (negedge CSn, posedge CLK, tCSS,  tCSS_notifier);
    $hold  (negedge CLK, posedge CSn, tCSH,  tCSH_notifier);

    $width (posedge CLK, tCH, 0, tCH_notifier); 
    $width (negedge CLK, tCL, 0, tCL_notifier);
    $width (posedge CSn, tCS, 0, tCS_notifier);

    $setup (SI_IO0, posedge CLK, tSU,  tSU_notifier);
    $hold  (posedge CLK, SI_IO0, tHD,  tHD_notifier);

    $hold  (negedge CLK, SO_IO1, tOH,  tOH_notifier);

  endspecify 

  /* States */
  parameter S_IDLE     = 8'd0;
  parameter S_CMD      = 8'd1;
  parameter S_ADDR     = 8'd2;
  parameter S_DATAIN   = 8'd3;
  parameter S_DATAOUT  = 8'd4;
  parameter S_LATENCY  = 8'd5;
  parameter S_MODE     = 8'd6;
  parameter SD_ADDR    = 8'd14; // DDR 
  parameter SD_MODE    = 8'd15; // DDR 
  parameter SD_DATAIN  = 8'd16; // DDR
  parameter SD_DATAOUT = 8'd17; // DDR
  parameter S_IGNORE   = 8'hff;
  reg [7:0] state;
  reg s_cmd_done;
  reg s_addr_done;
  reg s_datain_done;
  reg s_dataout_done;
  reg s_latency_done;
  reg s_mode_done;
  reg modesw_valid;

  reg PORn;
  reg [7:0] mode;
  reg si_dq0_reg; 
  reg so_dq1_reg;
  reg wp_dq2_reg;
  reg hold_dq3_reg; 
  reg [3:0] dqout_en;
  reg INTn_wire;

  /* Address */
  reg [23:0] addr;
  wire [23:0] sr_addr;
  wire [23:0] sr_addr_ddr;
  localparam ADDR_SR  = 24'h0;
  localparam ADDR_CR1 = 24'h2;
  localparam ADDR_CR2 = 24'h3;
  localparam ADDR_INTCR = 24'h4;
  localparam ADDR_ECCDIR = 24'h5;
  localparam ADDR_ECCEIR = 24'h6;
  localparam ADDR_ECCDOR = 24'h7;
  localparam ADDR_ECCECR = 24'h8;
  localparam ADDR_FSR = 24'hA;
  localparam ADDR_ID  = 24'h30;

  /* Device config/status registers */
  reg [7:0] SR;
  localparam SR_WRMASK = 8'hFD;
  localparam SR_MAPLK  = 8'hC1;
  localparam SR_DEFAULT = 8'h00;
  reg [7:0] CR1, CR2;
  localparam CR1_WRMASK = 8'hE7;
  localparam CR2_WRMASK = 8'h1F;
  localparam CR1_DEFAULT = 8'h60;
  localparam CR2_DEFAULT = 8'h08;
  wire [31:0] IDREG;
`ifdef DENSITY_64M
  assign IDREG = {8'hE6, 4'h2, 4'h1, 4'h2, 4'h1, 8'h1};
`elsif DENSITY_128M
  assign IDREG = {8'hE6, 4'h2, 4'h1, 4'h2, 4'h2, 8'h1};
`else
  assign IDREG = {8'hE6, 4'h2, 4'h1, 4'h2, 4'h2, 8'h1};
`endif

  reg [7:0] FSR;
  localparam FSR_DEFAULT = 8'h80;

  reg [7:0] INTCR;
  localparam INTCR_WRMASK = 8'h63;
  localparam INTCR_DEFAULT = 8'h00;

  reg [31:0] ECCDIR;
  localparam ECCDIR_DEFAULT = 32'h0;

  reg [31:0] ECCEIR;
  localparam ECCEIR_DEFAULT = 32'h0;

  reg [31:0] ECCDOR;
  reg [31:0] ECCECR;

  wire wren;
  wire [2:0] bpsel;
  wire tbsel;
  wire wpnen;
  assign wren = SR[1];
  assign bpsel = SR[4:2];
  assign tbsel = SR[5];
  assign wpnen = SR[7];
  wire maplk;
  assign maplk = CR1[2];
  wire [3:0] mlats;
  assign mlats = CR2[3:0];
  wire [2:0] odsel;
  assign odsel = CR1[7:5];
  wire [1:0] wrens;
  assign wrens = CR1[1:0];
  wire xip_write;
  assign xip_write = CR2[4];

  assign INTn = INTn_wire;
  always@(negedge INTn)
    $display("[MODEL] Interrupt Asserted");
 

  wire wr_allow; // WREN + WRENS logic
  assign wr_allow = (((wrens == 2'b00) && wren) ||
                     (wrens == 2'b01) ||
                     ((wrens == 2'b10) && wren)
                    )? 1'b1 : 0;


  /* Memory Array */
  localparam ARRAY_BYTES = 24'h800000;
  localparam UPPER_1_64 = 24'h7E0000;
  localparam UPPER_1_32 = 24'h7C0000;
  localparam UPPER_1_16 = 24'h780000;
  localparam UPPER_1_8  = 24'h700000;
  localparam UPPER_1_4  = 24'h600000;
  localparam UPPER_1_2  = 24'h400000;
  localparam LOWER_1_64 = 24'h01FFFF;
  localparam LOWER_1_32 = 24'h03FFFF;
  localparam LOWER_1_16 = 24'h07FFFF;
  localparam LOWER_1_8  = 24'h0FFFFF;
  localparam LOWER_1_4  = 24'h1FFFFF;
  localparam LOWER_1_2  = 24'h3FFFFF;

/*SCK period*/
  real t0, t1;
  real SCK_period;

  always@(negedge CSn) begin
    @(posedge CLK) t0 = $realtime;
    @(posedge CLK) t1 = $realtime;
    SCK_period = t1 - t0;
  end

/*CSn JEDEC*/
  real c0, c1;
  real CSn_period;

  always@(negedge CSn) begin
    c0 = $realtime;
    @(posedge CSn) c1 = $realtime;
    CSn_period = c1 - c0;
  end

  reg [7:0] mem_array [0:ARRAY_BYTES-1];
  reg [23:0] wr_addr, rd_addr;

  /* Address protect */
  reg addr_prot;
  always@(*)
    if (wr_addr >= ARRAY_BYTES) begin // Above capacity 
      addr_prot <= 1'b1;
    end
    else begin
      if ((tbsel | HTBSEL) == 0) // Top Block Protection
        case(bpsel | HBP)
          3'b001: addr_prot <= (wr_addr >= UPPER_1_64)? 1'b1 : 0;
          3'b010: addr_prot <= (wr_addr >= UPPER_1_32)? 1'b1 : 0;
          3'b011: addr_prot <= (wr_addr >= UPPER_1_16)? 1'b1 : 0;
          3'b100: addr_prot <= (wr_addr >= UPPER_1_8)? 1'b1 : 0;
          3'b101: addr_prot <= (wr_addr >= UPPER_1_4)? 1'b1 : 0;
          3'b110: addr_prot <= (wr_addr >= UPPER_1_2)? 1'b1 : 0;
          3'b111: addr_prot <= 1'b1; 
          default: addr_prot <= 0;
        endcase
      else            // Bottom Block Protection
        case(bpsel | HBP)
          3'b001: addr_prot <= (wr_addr <= LOWER_1_64)? 1'b1 : 0;
          3'b010: addr_prot <= (wr_addr <= LOWER_1_32)? 1'b1 : 0;
          3'b011: addr_prot <= (wr_addr <= LOWER_1_16)? 1'b1 : 0;
          3'b100: addr_prot <= (wr_addr <= LOWER_1_8)? 1'b1 : 0;
          3'b101: addr_prot <= (wr_addr <= LOWER_1_4)? 1'b1 : 0;
          3'b110: addr_prot <= (wr_addr <= LOWER_1_2)? 1'b1 : 0;
          3'b111: addr_prot <= 1'b1; 
          default: addr_prot <= 0;
        endcase
    end

  /* SPI Pin write protect */
  reg hw_wp;
  always@(*)
    if (wpnen && (mode == SPIEN))
      if (!WPn_IO2)
        hw_wp <= 1'b1;
      else
        hw_wp <= 0;
    else
      hw_wp <= 0;

  /* Latches */
  reg [7:0] cmd_latch;
  reg [7:0] mode_latch; // mode byte
  wire [7:0] datain_reg_ser;
  wire [7:0] datain_reg_dual;
  wire [7:0] datain_reg_quad;
  reg  [7:0] datain_reg;
  srdin_ser DATAIN_LATCH_SERIAL(
    .sclk (CLK),
    .cs_n (CSn),
    .din  (SI_IO0),
    .dout (datain_reg_ser)
    );

  srdin_dual DATAIN_LATCH_DUAL(
    .sclk (CLK),
    .cs_n (CSn),
    .din  ({SO_IO1, SI_IO0}),
    .dout (datain_reg_dual)
    );

  srdin_quad DATAIN_LATCH_QUAD(
    .sclk (CLK),
    .cs_n (CSn),
    .din  ({IO3, WPn_IO2, SO_IO1, SI_IO0}),
    .dout (datain_reg_quad)
    );

  wire [7:0] datain_reg_ddr_ser;
  wire [7:0] datain_reg_ddr_dual;
  wire [7:0] datain_reg_ddr_quad;
  srdin_ser_ddr DATAIN_LATCH_SERIAL_DDR(
    .sclk (CLK),
    .cs_n (CSn),
    .din  (SI_IO0),
    .dout (datain_reg_ddr_ser)
    );

  srdin_dual_ddr DATAIN_LATCH_DUAL_DDR(
    .sclk (CLK),
    .cs_n (CSn),
    .din  ({SO_IO1, SI_IO0}),
    .dout (datain_reg_ddr_dual)
    );

  srdin_quad_ddr DATAIN_LATCH_QUAD_DDR(
    .sclk (CLK),
    .cs_n (CSn),
    .din  ({IO3, WPn_IO2, SO_IO1, SI_IO0}),
    .dout (datain_reg_ddr_quad)
    );

  always@(*)
    case(state)
      SD_ADDR, SD_DATAIN:
        case(mode)
          SPIEN: begin
            case(cmd_latch)
              DWQO:
                datain_reg <= datain_reg_ddr_quad;
              default:
                datain_reg <= datain_reg_ddr_ser;
            endcase
          end
          QPIEN:   datain_reg <= datain_reg_ddr_quad;
          default: datain_reg <= 0;
        endcase
        
      S_MODE: // XIP mode byte input width
        case(mode)
          SPIEN: begin
            case(cmd_latch)
              RDQI, WQIO:
                datain_reg <= datain_reg_quad;
              default:
                datain_reg <= datain_reg_ser;
            endcase
          end
          QPIEN:   datain_reg <= datain_reg_quad;
          default: datain_reg <= 0;
        endcase

      SD_MODE: // XIP mode byte input width
        case(mode)
          SPIEN: begin
            case(cmd_latch)
              DRQI, DWQO:
                datain_reg <= datain_reg_ddr_quad;
              default:
                datain_reg <= datain_reg_ddr_ser;
            endcase
          end
          QPIEN:   datain_reg <= datain_reg_ddr_quad;
          default: datain_reg <= 0;
        endcase

      S_DATAIN:
        case(mode)
          SPIEN: begin
            case(cmd_latch)
              RDQI, WQIO:
                datain_reg <= datain_reg_quad;
              default:
                datain_reg <= datain_reg_ser;
            endcase
          end
          QPIEN:   datain_reg <= datain_reg_quad;
          default: datain_reg <= 0;
        endcase

      default:
        case(mode)
          SPIEN:   datain_reg <= datain_reg_ser;
          QPIEN:   datain_reg <= datain_reg_quad;
          default: datain_reg <= 0;
        endcase
    endcase

  reg [2:0]  addr_mode_sel;
  addr_reg ADDR_LATCH(
    .sclk     (CLK),
    .cs_n     (CSn),
    .mode_sel (addr_mode_sel),
    .din      ({IO3, WPn_IO2, SO_IO1, SI_IO0}),
    .addr     (sr_addr)
    );

  addr_reg_ddr ADDR_LATCH_DDR(
    .sclk     (CLK),
    .cs_n     (CSn),
    .mode_sel (addr_mode_sel),
    .din      ({IO3, WPn_IO2, SO_IO1, SI_IO0}),
    .addr     (sr_addr_ddr)
    );


  always@(posedge CLK)
    if (s_addr_done) begin
      if (state == SD_ADDR)
        addr <= sr_addr_ddr;
      else
        addr <= sr_addr;
    end

  always@(*)
    case(mode)
      SPIEN: begin
        case(cmd_latch)
          RDQI, DRQI, WQIO, DWQO:
            addr_mode_sel <= 3'b100;
          default:
            addr_mode_sel <= 3'b001;
        endcase
      end
      QPIEN: begin
	addr_mode_sel <= 3'b100;
      end
      default: begin
	addr_mode_sel <= 3'b000;
      end
    endcase

  /* XIP mode enable */
  reg xip_en;

  /* SCLK counter */
  reg [5:0] sclk_re_ctr;

  always@(posedge CLK or negedge PORn or posedge CSn or negedge RESETn)
    if (!PORn || CSn || !RESETn)
      sclk_re_ctr <= 0;
    else
      case(state)
        S_CMD:
          if (s_cmd_done)
            sclk_re_ctr <= 0;
          else
            sclk_re_ctr <= sclk_re_ctr + 1;

        S_ADDR, SD_ADDR:
          if (s_addr_done)
            sclk_re_ctr <= 0;
          else
            sclk_re_ctr <= sclk_re_ctr + 1; 

        S_MODE, SD_MODE:
          if (s_mode_done)
            sclk_re_ctr <= 0;
          else
            sclk_re_ctr <= sclk_re_ctr + 1; 

        S_DATAIN, S_DATAOUT, SD_DATAIN, SD_DATAOUT: 
          if (s_datain_done || s_dataout_done)
            sclk_re_ctr <= 0;
          else
            sclk_re_ctr <= sclk_re_ctr + 1;

        S_LATENCY:
          if (s_latency_done)
            sclk_re_ctr <= 0;
          else
            sclk_re_ctr <= sclk_re_ctr + 1;

        default:
          sclk_re_ctr <= 0;
      endcase

  /*
   * Byte strobe
   */
  reg byte_strobe;
  reg [7:0] byte_strobe_ctr;
  always@(*)
    case(mode)
      SPIEN:
        case(sclk_re_ctr)
          6'd7, 6'd15, 6'd23, 6'd31, 6'd39, 6'd47, 6'd55, 6'd63:
            byte_strobe = 1'b1;
          6'd3, 6'd11, 6'd19, 6'd27, 6'd35, 6'd43, 6'd51, 6'd59:
            case(cmd_latch)
              DRFR, DRFW:
                byte_strobe = 1'b1;
              default:
                byte_strobe = 0;
            endcase
          6'd1:
            case(cmd_latch)
              RDQO, RDQI, WQIO, RDQO1:
                byte_strobe = 1'b1;
              default:
                byte_strobe = 0;
            endcase
          6'd0:
            case(cmd_latch)
              DRQI, DWQO:
                byte_strobe = 1'b1;
              default:
                byte_strobe = 0;
            endcase
          default:
            byte_strobe = 0;
        endcase

      QPIEN:
        case(sclk_re_ctr)
          6'd1, 6'd3, 6'd5, 6'd7, 6'd9, 6'd11, 6'd13, 6'd15:
            byte_strobe = 1'b1;
          6'd0, 6'd2, 6'd4, 6'd6, 6'd8, 6'd10, 6'd12, 6'd14:
            case(cmd_latch)
              DRFR, DRFW:
                byte_strobe = 1'b1;
              default:
                byte_strobe = 0;
            endcase
          default:
            byte_strobe = 0;
        endcase

      default:
        byte_strobe = 0;
    endcase

  always@(posedge CLK)
    case(state)
      S_DATAIN, S_DATAOUT, SD_DATAIN, SD_DATAOUT:
        if (byte_strobe)
          byte_strobe_ctr = byte_strobe_ctr + 1;
      default:
        byte_strobe_ctr = 0;
    endcase

  // Power on reset
  integer i;
  initial
  begin
    /* Initialize Array */
    for (i=0; i<ARRAY_BYTES; i=i+1) begin
      mem_array[i] = 0;
    end
    PORn = 1'b0;
    #tPOR_time PORn = 1'b1;
    mode = SPIEN;
  end

  // Initial Reset
  always @(posedge PORn or posedge RESETn)
  begin
    hold_dq3_reg = 0; 
    si_dq0_reg = 0; 
    so_dq1_reg = 0;
    wp_dq2_reg = 0;
  end

  /* Output pin assignment */
  assign #(tCO) SI_IO0  = (dqout_en[0]) ? si_dq0_reg   : 1'bz;
  assign #(tCO) SO_IO1  = (dqout_en[1]) ? so_dq1_reg   : 1'bz;
  assign #(tCO) WPn_IO2 = (dqout_en[2]) ? wp_dq2_reg   : 1'bz;
  assign #(tCO) IO3     = (dqout_en[3]) ? hold_dq3_reg : 1'bz;

  reg [7:0] dataout_reg;
  reg dataout_reg_shift;
  reg dataout_reg_latch;
  
  wire dataout_reg_ser_dout;
  wire dataout_reg_ser_ddr;
  wire [1:0] dataout_reg_dual_dout;
  wire [1:0] dataout_reg_dual_ddr;
  wire [3:0] dataout_reg_quad_dout;
  wire [3:0] dataout_reg_quad_ddr;

  srdout_ser DATAOUT_LATCH_SERIAL(
    .sclk  (CLK),
    .cs_n  (CSn),
    .rstn  (RESETn),
    .latch (dataout_reg_latch),
    .shift (dataout_reg_shift),
    .din   (dataout_reg),
    .dout  (dataout_reg_ser_dout),
    .ddr   (dataout_reg_ser_ddr)
    );

  srdout_dual DATAOUT_LATCH_DUAL(
    .sclk  (CLK),
    .cs_n  (CSn),
    .rstn  (RESETn),
    .latch (dataout_reg_latch),
    .shift (dataout_reg_shift),
    .din   (dataout_reg),
    .dout  (dataout_reg_dual_dout),
    .ddr   (dataout_reg_dual_ddr)
    );

  srdout_quad DATAOUT_LATCH_QUAD(
    .sclk  (CLK),
    .cs_n  (CSn),
    .rstn  (RESETn),
    .latch (dataout_reg_latch),
    .shift (dataout_reg_shift),
    .din   (dataout_reg),
    .dout  (dataout_reg_quad_dout),
    .ddr   (dataout_reg_quad_ddr)
    );

  /* Data out register to pin output */
  always@(*)
    case(mode)
      SPIEN: begin
        case(cmd_latch)
          DRFR:
            so_dq1_reg = dataout_reg_ser_ddr;
          DRQI:
          begin
            si_dq0_reg   = dataout_reg_quad_ddr[0];
            so_dq1_reg   = dataout_reg_quad_ddr[1];
            wp_dq2_reg   = dataout_reg_quad_ddr[2];
            hold_dq3_reg = dataout_reg_quad_ddr[3];
          end
          RDQO, RDQI, RDQO1:
          begin
            si_dq0_reg   = dataout_reg_quad_dout[0];
            so_dq1_reg   = dataout_reg_quad_dout[1];
            wp_dq2_reg   = dataout_reg_quad_dout[2];
            hold_dq3_reg = dataout_reg_quad_dout[3];
          end
          default:
            so_dq1_reg = dataout_reg_ser_dout;
        endcase
      end

      QPIEN: begin
        case(cmd_latch)
          DRFR: begin
            si_dq0_reg   = dataout_reg_quad_ddr[0];
            so_dq1_reg   = dataout_reg_quad_ddr[1];
            wp_dq2_reg   = dataout_reg_quad_ddr[2];
            hold_dq3_reg = dataout_reg_quad_ddr[3];
          end
          default: begin
            si_dq0_reg   = dataout_reg_quad_dout[0];
            so_dq1_reg   = dataout_reg_quad_dout[1];
            wp_dq2_reg   = dataout_reg_quad_dout[2];
            hold_dq3_reg = dataout_reg_quad_dout[3];
          end
        endcase
      end
    endcase

  always@(negedge CLK or posedge CSn)
    if (CSn)
      dqout_en = 0;
    else begin
      dqout_en = 0;
      case(state)
        S_MODE, SD_MODE: // Special case for MLATS = 0
          if ((mlats == 0) && s_mode_done) begin
            case(cmd_latch)
              RDFT, DRFR, RDFT1:
                case(mode)
                  SPIEN: dqout_en = 4'b0010;
                  QPIEN: dqout_en = 4'b1111;
                endcase

              RDQO, RDQI, DRQI, RDQO1:
                dqout_en = 4'b1111;

              default:
                dqout_en = 0;
            endcase
          end
          else
            dqout_en = 0;

        S_ADDR:
          if (s_addr_done)
            case(cmd_latch)
              READ, READ1: // Restricted to 1-1-1
                case(mode)
                  SPIEN: dqout_en = 4'b0010;
                  //QPIEN: dqout_en = 4'b1111;
                  default: dqout_en = 0;
                endcase

              RDAR:
                  dqout_en = 0;

              default:
                dqout_en = 0;
            endcase
          
        S_DATAOUT, SD_DATAOUT:
          case(cmd_latch)
            READ, READ1: //Restricted to 1-1-1
              case(mode)
                SPIEN: dqout_en = 4'b0010;
                //QPIEN: dqout_en = 4'b1111;
                default: dqout_en = 0;
              endcase

            RDFT, DRFR, RDFT1:
              case(mode)
                SPIEN: dqout_en = 4'b0010;
                QPIEN: dqout_en = 4'b1111;
                default: dqout_en = 0;
              endcase

            RDQO, RDQI, DRQI, RDQO1:
              dqout_en = 4'b1111;

            default: 
              if (!s_dataout_done)
                case(mode)
                  SPIEN: dqout_en = 4'b0010;
                  QPIEN: dqout_en = 4'b1111;
                  default: dqout_en = 0;
                endcase
          endcase

        S_CMD:
          if (s_cmd_done)
            case(datain_reg)
              RDSR, RDID: 
                case(mode)
                  SPIEN: dqout_en = 4'b0010;
		  QPIEN: dqout_en = 4'b1111;
                  default: dqout_en = 0;
                endcase
              RDFSR: 
                case(mode)
                  SPIEN: dqout_en = 4'b0010;
                  QPIEN: dqout_en = 4'b1111;
                  default: dqout_en = 0;
                endcase
            endcase
          else
            dqout_en = 0;
            
        S_LATENCY:
          if (s_latency_done)
            case(cmd_latch) 
              RDQO, RDQI, DRQI, RDQO1:
                dqout_en = 4'b1111;

              default:
                case(mode)
                  SPIEN: dqout_en = 4'b0010;
                  //DPIEN: dqout_en = 4'b0011;
                  QPIEN: dqout_en = 4'b1111;
                  default: dqout_en = 0;
                endcase
            endcase

          else
            dqout_en = 0;

      endcase
    end

  /* State machine */
  always@(posedge CLK or negedge PORn or posedge CSn or negedge CSn or negedge RESETn)

    if (CSn) begin
      state <= S_IDLE;
    end
    else if (CLK) begin
      case (state)
        S_IDLE:
          if (xip_en) begin /* XIP transition direct to Addr states */
            case(cmd_latch)
              DRFR, DRQI:
                state <= SD_ADDR;
              DRFW, DWQO:
                if ((wrens == 2'b00) && wren)
                  state <= SD_ADDR;
                else if ((wrens == 2'b01) || (wrens == 2'b10))
                  state <= SD_ADDR;
                else begin
                  state <= S_IGNORE;
                  $display("[MODEL] WREN not set. Write Ignored (XIP)");
                end
              WRFT, WQIO:
                if ((wrens == 2'b00) && wren)
                  state <= S_ADDR;
                else if ((wrens == 2'b01) || (wrens == 2'b10))
                  state <= S_ADDR;
                else begin
                  state <= S_IGNORE;
                  $display("[MODEL] WREN not set. Write Ignored (XIP)");
                end
              default:
                state <= S_ADDR;
            endcase
          end
          else begin
            state <= S_CMD;
          end

        S_CMD:
          if (s_cmd_done)
            case(cmd_latch)
              RDSR, RDID:
                  state <= S_DATAOUT;

              RDFSR:
                  state <= S_DATAOUT;

              WRSR:
                  state <= S_DATAIN;

              READ:
                if (mode == SPIEN) begin
                  state <= S_ADDR;
                end
                else begin
                  state <= S_IGNORE;
                  $display("[MODEL] READ ignored (restricted to SPI)");
                end

              READ1:
                if (mode == SPIEN) begin
                  state <= S_ADDR; //4byte address
                end
                else begin
                  state <= S_IGNORE;
                  $display("[MODEL] READ ignored (restricted to SPI)");
                end

              RDFT:
                state <= S_ADDR;

              RDFT1:
                state <= S_ADDR;

              RDAR, WRAR:
                  state <= S_ADDR;
        
              RDQO, RDQI:
                if (mode == SPIEN)
                  state <= S_ADDR;
                else begin
                  state <= S_IGNORE;
                  $display("[MODEL] RDQO/RDQI ignored (restricted to SPI)");
                end

              RDQO1:
                if (mode == SPIEN)
                  state <= S_ADDR;
                else begin
                  state <= S_IGNORE;
                  $display("[MODEL] RDQO ignored (restricted to SPI)");
                end

                
              WRTE:
                if ((mode == SPIEN) || (mode == QPIEN)) begin
                  if (wr_allow) begin
                    state <= S_ADDR;
                  end
                  else begin
                    if ((wrens == 2'b00) || (wrens == 2'b10)) begin
                      $display("[MODEL] WREN not set. WRTE ignored");
                      state <= S_IGNORE;
                    end
                    else begin
                      $display("[MODEL] Invalid WRENS value 0x%X. WRTE ignored", wrens);
                      state <= S_IGNORE;
                    end
                  end
                end
                else begin
                  state <= S_IGNORE;
                  $display("[MODEL] WRTE ignored (restricted to SPI/QPI)");
                end

              WRFT, WQIO:
                if (wr_allow) begin
                  state <= S_ADDR;
                end
                else begin
                  state <= S_IGNORE;
                  if ((wrens == 2'b00) || (wrens == 2'b10)) begin
                    $display("[MODEL] WREN not set. Write ignored");
                  end
                  else begin
                    $display("[MODEL] Invalid WRENS value 0x%X. Write ignored", wrens);
                  end
                end

              DRFR, DRQI:
                state <= SD_ADDR;

              DRFW, DWQO:
                if (wr_allow) begin
                  state <= SD_ADDR;
                end
                else begin
                  state <= S_IGNORE;
                  if ((wrens == 2'b00) || (wrens == 2'b10)) begin
                    $display("[MODEL] WREN not set. Write ignored");
                  end
                  else begin
                    $display("[MODEL] Invalid WRENS value 0x%X. DRFW ignored", wrens);
                  end
                end

              default:
                state <= S_IGNORE;
            endcase

        S_ADDR:
          if (s_addr_done)
            case(cmd_latch)
              RDAR:
                if (mlats == 0)
                  state <= S_DATAOUT;
                else
                  state <= S_LATENCY;
              WRAR:  begin
                state <= S_DATAIN;
                case(sr_addr)
                  ADDR_SR, ADDR_CR1, ADDR_CR2, ADDR_INTCR,
                  ADDR_ECCDIR, ADDR_ECCEIR:
                    $display("[MODEL] WRAR address valid");
                  ADDR_ID, ADDR_ECCDOR, ADDR_ECCECR, ADDR_FSR:
                    $display("[MODEL] WRAR address not written (Read-only register)");
                  default:
                    $display("[MODEL] WRAR address invalid: 0x%x", sr_addr);
                endcase
              end
              WRTE:
                state <= S_DATAIN;
              READ, READ1:
                state <= S_DATAOUT;
              WRFT, WQIO:
                  if(!xip_write)
                   state <= S_MODE;
                  else
                   state <= S_DATAIN;
	      RDQI:
                state <= S_MODE;
              RDFT, RDFT1, RDQO, RDQO1:
                state <= S_LATENCY;
              default:
                state <= S_IGNORE;
            endcase

        SD_ADDR:
          if (s_addr_done)
            case(cmd_latch)
              DRFW, DWQO:
                  if(!xip_write)
                   state <= SD_MODE;
                  else
                   state <= SD_DATAIN;
	      DRFR, DRQI:
                state <= SD_MODE;
              default:
                state <= S_IGNORE;
            endcase

        S_MODE:
          if (s_mode_done)
            case(cmd_latch)
              WRFT, WQIO:
                state <= S_DATAIN;
              RDFT, RDQO, RDQI, RDFT1, RDQO1:
                if (mlats != 0)
                  state <= S_LATENCY;
                else
                  state <= S_DATAOUT;
              default:
                state <= S_IGNORE;
            endcase

        SD_MODE:
          if (s_mode_done)
            case(cmd_latch)
              DRFR, DRQI:
                if (mlats != 0)
                  state <= S_LATENCY;
                else
                  state <= SD_DATAOUT;
              DRFW, DWQO:
                state <= SD_DATAIN;
              default:
                state <= S_IGNORE;
            endcase

        S_LATENCY:
          if (s_latency_done)
            case(cmd_latch)
              RDAR, RDFT, RDQO, RDQI, RDFT1, RDQO1:
                state <= S_DATAOUT;
              DRFR, DRQI:
                state <= SD_DATAOUT;
              default:
                state <= S_IGNORE;
            endcase

        S_DATAIN:
          if (s_datain_done)
            case(cmd_latch)
              WRSR, WRAR:
                state <= S_IDLE;
              WRTE, WRFT, WQIO:
                state <= S_DATAIN; // continue until CSn deasserted
              default:
                state <= S_IGNORE;
            endcase

        SD_DATAIN:
          if (s_datain_done)
            case(cmd_latch)
              DRFW, DWQO:
                state <= SD_DATAIN; // continue until CSn deasserted
              default:
                state <= S_IGNORE;
            endcase

        S_DATAOUT:
          if (s_dataout_done)
            case(cmd_latch)
              RDSR, RDAR, RDID, RDFSR :
                state <= S_IDLE;
              READ, RDFT, RDQO, RDQI, READ1, RDFT1, RDQO1:
                state <= S_DATAOUT; // continue until CSn deasserted
              default:
                state <= S_IGNORE;
            endcase

        SD_DATAOUT:
          case(cmd_latch)
            DRFR, DRQI:
              state <= SD_DATAOUT; // continue until CSn deasserted
            default:
              state <= S_IGNORE;
          endcase

        S_IGNORE:
          state <= S_IGNORE;

        default:
          state <= S_IDLE;
      endcase
    end

  /* Command Notifier */
  always@(negedge CLK)
    if (s_cmd_done)
      case(datain_reg)
        WREN: $display("[MODEL] Command received 0x%x (WREN)", datain_reg);
        WRDI: $display("[MODEL] Command received 0x%x (WRDI)", datain_reg);
        RDSR:
          if (mode == SPIEN) begin
            $display("[MODEL] Command received 0x%x (RDSR)", datain_reg);
          end
        WRSR:
          if (mode == SPIEN) begin
            $display("[MODEL] Command received 0x%x (WRSR)", datain_reg);
          end
        SPIE: 
          if (modesw_valid) 
            $display("[MODEL] Command received 0x%x (SPIE)", datain_reg);
          else
            $display("[MODEL] Command received 0x%x (SPIE), but already in SPI mode", datain_reg);
        QPIE: 
          if (modesw_valid)
            $display("[MODEL] Command received 0x%x (QPIE)", datain_reg);
          else
            $display("[MODEL] Command received 0x%x (QPIE), but already in QPI mode", datain_reg);

        RDFSR: $display("[MODEL] Command received 0x%x (RDFSR)", datain_reg);
        RDAR: $display("[MODEL] Command received 0x%x (RDAR)", datain_reg);
        WRAR: $display("[MODEL] Command received 0x%x (WRAR)", datain_reg);
        WRTE: $display("[MODEL] Command received 0x%x (WRTE)", datain_reg);
        READ, READ1:
          $display("[MODEL] Command received 0x%x (READ)", datain_reg);
        WRFT: $display("[MODEL] Command received 0x%x (WRFT)", datain_reg);
        RDFT, RDFT1: 
          $display("[MODEL] Command received 0x%x (RDFT)", datain_reg);
        RDID:
          if (mode == SPIEN) begin
            $display("[MODEL] Command received 0x%x (RDID)", datain_reg);
          end
        DRFR: $display("[MODEL] Command received 0x%x (DRFR)", datain_reg);
        DRFW: $display("[MODEL] Command received 0x%x (DRFW)", datain_reg);
        RDQO, RDQO1: 
          $display("[MODEL] Command received 0x%x (RDQO)", datain_reg);
        RDQI: $display("[MODEL] Command received 0x%x (RDQI)", datain_reg);
        WQIO: $display("[MODEL] Command received 0x%x (WQIO)", datain_reg);
        DRQI: $display("[MODEL] Command received 0x%x (DRQI)", datain_reg);
        DWQO: $display("[MODEL] Command received 0x%x (DWQO)", datain_reg);
        default: $display("[MODEL] Unsupported command 0x%x", datain_reg);
      endcase

  /* Dataout cycles LUT */
  reg [7:0] s_dataout_cc;
  always@(*)
    case(cmd_latch)
      RDAR:
        case(addr)
          //ADDR_SR, ADDR_CR1, ADDR_CR2, ADDR_CR3, ADDR_CR4:
          ADDR_SR, ADDR_CR1, ADDR_CR2, ADDR_INTCR, ADDR_FSR:
            s_dataout_cc <= 8'd7; // 8-1
          ADDR_ID, ADDR_ECCDIR, ADDR_ECCEIR, ADDR_ECCDOR, ADDR_ECCECR:
            s_dataout_cc <= 8'd31; // 32-1
                   
          default:
            s_dataout_cc <= 8'd7; // 8-1
        endcase

      default:
       s_dataout_cc <= 0;
    endcase

  /* State done signals */
  always@(*)
    if (state == S_CMD)
      case(mode)
        SPIEN: s_cmd_done <= (sclk_re_ctr == 6'd7)? 1'b1 : 0;
        QPIEN: s_cmd_done <= (sclk_re_ctr == 6'd1)? 1'b1 : 0;
        default: s_cmd_done <= 0;
      endcase
    else
      s_cmd_done <= 0;

  always@(*)
    if (state == S_ADDR)
      case(mode)
        SPIEN: begin
          case(cmd_latch)
            RDQI, WQIO:
              s_addr_done <= (sclk_re_ctr == 6'd5)? 1'b1 : 0; 
            default:
              s_addr_done <= (sclk_re_ctr == 6'd23)? 1'b1 : 0; 
          endcase
        end
        QPIEN: s_addr_done <= (sclk_re_ctr == 6'd5) ? 1'b1 : 0; 
        default: s_addr_done <= 0;
      endcase
    else if (state == SD_ADDR)
      case(mode)
        SPIEN: begin
          case(cmd_latch)
            DRQI, DWQO:
              s_addr_done <= (sclk_re_ctr == 6'd2)? 1'b1 : 0;
            default:
              s_addr_done <= (sclk_re_ctr == 6'd11)? 1'b1 : 0;
          endcase
        end
        QPIEN: s_addr_done <= (sclk_re_ctr == 6'd2) ? 1'b1 : 0;
        default: s_addr_done <= 0;
      endcase
    else
      s_addr_done <= 0;

  always@(*)
    if (state == S_MODE)
      case(mode)
        SPIEN: begin 
          case(cmd_latch)
            RDQI, WQIO:
              s_mode_done <= (sclk_re_ctr == 6'd1)? 1'b1 : 0;
            default:
              s_mode_done <= (sclk_re_ctr == 6'd7)? 1'b1 : 0;
          endcase
        end
        QPIEN: s_mode_done <= (sclk_re_ctr == 6'd1)? 1'b1 : 0;
        default: s_mode_done <= 0;
      endcase
    else if (state == SD_MODE)
      case(mode)
        SPIEN: begin
          case(cmd_latch)
            DRQI, DWQO:
              s_mode_done <= 1'b1; //1cc done
            default:
              s_mode_done <= (sclk_re_ctr == 6'd3)? 1'b1 : 0;
          endcase
        end
        QPIEN: s_mode_done <= (sclk_re_ctr == 6'd0)? 1'b1 : 0;
        default: s_mode_done <= 0;
      endcase
    else
      s_mode_done <= 0;

  always@(*)
    if ((state == S_DATAIN) || (state == SD_DATAIN))
      case(cmd_latch)
        WRSR, WRTE, WRFT: 
          case(mode)
            SPIEN: s_datain_done <= (sclk_re_ctr == 6'd7)? 1'b1 : 0;
            QPIEN: s_datain_done <= (sclk_re_ctr == 6'd1)? 1'b1 : 0;
            default: s_datain_done <= 0;
          endcase

        WRAR:
          case(addr)
            ADDR_ECCDIR, ADDR_ECCEIR:
	     begin
	      if(mode == SPIEN)
               s_datain_done <= (sclk_re_ctr == 6'd31)? 1'b1 : 0;
	      else
               s_datain_done <= (sclk_re_ctr == 6'd7)? 1'b1 : 0;
	     end
            default:
	     begin
	      if(mode == SPIEN)
               s_datain_done <= (sclk_re_ctr == 6'd7)? 1'b1 : 0;
	      else
               s_datain_done <= (sclk_re_ctr == 6'd1)? 1'b1 : 0;
	     end
          endcase

        DRFW, WQIO, DWQO:
          s_datain_done <= byte_strobe;

        default:
          s_datain_done <= 0;
      endcase
    else
      s_datain_done <= 0;

  always@(*)
    case(state)
      S_DATAOUT:
        case(cmd_latch)
          RDSR, READ, RDFT, READ1, RDFT1, RDFSR:
            case(mode)
              SPIEN: s_dataout_done <= (sclk_re_ctr == 6'd7)? 1'b1 : 0;
              QPIEN: s_dataout_done <= (sclk_re_ctr == 6'd1)? 1'b1 : 0;
              default: s_dataout_done <= 0;
            endcase

          RDQO, RDQI, RDQO1:
            s_dataout_done <= byte_strobe;

          RDID:
            case(mode)
              SPIEN: s_dataout_done <= (sclk_re_ctr == 6'd31)? 1'b1 : 0;
              default: s_dataout_done <= 0;
            endcase

          RDAR:
            case(mode)
              SPIEN: s_dataout_done <= (sclk_re_ctr == s_dataout_cc)? 1'b1 : 0;
              default: s_dataout_done <= 0;
            endcase

          default:
            s_dataout_done <= 0;
        endcase

      SD_DATAOUT:
        case(cmd_latch)
          DRFR, DRQI:
            s_dataout_done <= byte_strobe;

          default:
            s_dataout_done <= 0;
        endcase

      default:
        s_dataout_done <= 0;

    endcase

  always@(*)
    if (state == S_LATENCY)
      if (sclk_re_ctr == ({1'd0, mlats} - 6'd1))
        s_latency_done <= 1'b1;
      else
        s_latency_done <= 1'b0;
    else
      s_latency_done <= 0;

  /* Command Latch */
  always@(negedge CLK or negedge CSn)
    if (s_cmd_done) begin
      cmd_latch <= datain_reg;
    end

  /* Mode */
  always@(negedge CLK)
    case(state)
      S_CMD:
        if (s_cmd_done)
          case(datain_reg)
            SPIE:
              if (mode == QPIEN) begin
                mode = SPIEN;
                $display("[MODEL] Mode set to SPI");
              end

            QPIE:
              if (mode == SPIEN) begin
                mode = QPIEN;
                $display("[MODEL] Mode set to QPI");
              end

          endcase
    endcase

  /* Status Register
   * WREN bit may be cleared earlier than end of transaction since this is
   * just a model
   */
  always@(negedge CLK)
    if (!hw_wp) begin
      case(state)
        S_CMD:
          if (s_cmd_done) 
            case(datain_reg)
              WREN: begin
                SR <= SR | 8'h02;
                $display("[MODEL] WREN bit set. SR=0x%x", (SR | 8'h02));
              end
              WRDI: begin
                SR <= SR & 8'hFD;
                $display("[MODEL] WREN bit reset. SR=0x%x", (SR & 8'hFD));
              end
            endcase

        S_DATAIN, SD_DATAIN:
          if (s_datain_done)
            case(cmd_latch)

              WRTE, WRFT, DRFW, WQIO, DWQO:
              begin
                if ((wrens == 2'b00) && !xip_en) // Normal mode, XIP not enabled
                  SR <= SR & 8'hFD; // clear WREN
                // else, don't clear (for wrens == 2'b01 and 2'b10, Normal
                // mode XIP enabled)
              end

              WRSR: begin
                if (wren) begin
                  if (!maplk) begin
                    SR <= ((SR & ~SR_WRMASK) | (datain_reg & SR_WRMASK)) & 8'hFD; // clear WREN as well
                    $display("[MODEL] SR written. SR=0x%x", 
                      ((SR & ~SR_WRMASK) | (datain_reg & SR_WRMASK)) & 8'hFD);
                    end
                  else begin
                    SR <= ((SR & ~SR_MAPLK) | (datain_reg & SR_MAPLK)) & 8'hFD; // clear WREN as well
                    $display("[MODEL] SR written (MAPLK set). SR=0x%x", 
                      ((SR & ~SR_MAPLK) | (datain_reg & SR_MAPLK)) & 8'hFD);
                  end
                end
                else
                  $display("[MODEL] WREN bit not set. SR not written. SR=0x%x", SR);
              end

              WRAR: begin
                if (addr == ADDR_SR) begin
                  if (wren) begin
                    if (!maplk) begin
                      SR <= ((SR & ~SR_WRMASK) | (datain_reg & SR_WRMASK)) & 8'hFD; // clear WREN as well
                      $display("[MODEL] SR written. SR=0x%x", 
                        ((SR & ~SR_WRMASK) | (datain_reg & SR_WRMASK)) & 8'hFD);
                      end
                    else begin
                      SR <= ((SR & ~SR_MAPLK) | (datain_reg & SR_MAPLK)) & 8'hFD; // clear WREN as well
                      $display("[MODEL] SR written (MAPLK set). SR=0x%x", 
                        ((SR & ~SR_MAPLK) | (datain_reg & SR_MAPLK)) & 8'hFD);
                    end
                  end
                  else begin
                    $display("[MODEL] WREN bit not set. SR not written. SR=0x%x", SR);
                  end
                end
                else begin
                  SR <= SR & 8'hFD; // clear WREN
                end
              end

            endcase
        endcase
    end

  /* Mode switch valid */
  always@(*)
    case(mode)
      SPIE:
        case(datain_reg)
          QPIEN:        modesw_valid = 1'b1;
          default:      modesw_valid = 0;
        endcase
      QPIE:
        case(datain_reg)
          SPIEN:        modesw_valid = 1'b1;
          default:      modesw_valid = 0;
        endcase
    endcase

  /* Config Registers */
  always@(negedge CLK)
    if (!hw_wp) begin
      case(state)

        S_DATAIN:
          case(cmd_latch)
            WRAR: begin
              if (wren) begin
                if (s_datain_done) begin
                  case(addr)
                    ADDR_CR1: begin
                      CR1 <= (CR1 & ~CR1_WRMASK) | (datain_reg & CR1_WRMASK);
                      $display("[MODEL] CR1 written. CR1=0x%x",
                        (CR1 & ~CR1_WRMASK) | (datain_reg & CR1_WRMASK));
                    end
                    ADDR_CR2: begin
                      CR2 <= (CR2 & ~CR2_WRMASK) | (datain_reg & CR2_WRMASK);
                      $display("[MODEL] CR2 written. CR2=0x%x",
                        (CR2 & ~CR2_WRMASK) | (datain_reg & CR2_WRMASK));
                    end
                  endcase
                end
              end
              else begin
                if (s_datain_done)
                  case(addr)
                    ADDR_CR1, ADDR_CR2:
                      $display("[MODEL] WREN bit not set CRx not written");
                  endcase
              end

            end


          endcase
      endcase
    end

  /* Interrupt, Extended and ECC registers */
  always@(negedge CLK)
    if (!hw_wp) begin
      if ((state == S_DATAIN) && (cmd_latch == WRAR)) begin
        if (wren) begin
          if (s_datain_done | byte_strobe) begin
            case(addr)
              ADDR_INTCR: begin
                INTCR <= (INTCR & ~INTCR_WRMASK) | (datain_reg & INTCR_WRMASK);
                $display("[MODEL] INTCR written. INTCR=0x%x",
                  (INTCR & ~INTCR_WRMASK) | (datain_reg & INTCR_WRMASK));
              end
              ADDR_ECCDIR: begin
                case(byte_strobe_ctr)
                  8'd0: ECCDIR[31:24] <= datain_reg;
                  8'd1: ECCDIR[23:16] <= datain_reg;
                  8'd2: ECCDIR[15:8] <= datain_reg;
                  8'd3: begin
                    ECCDIR[7:0] <= datain_reg;
                    $display("[MODEL] ECCDIR written. ECCDIR=0x%x", 
                    {ECCDIR[31:8], datain_reg});
                  end
                endcase
              end
              ADDR_ECCEIR: begin
                case(byte_strobe_ctr)
                  8'd0: ECCEIR[31:24] <= datain_reg;
                  8'd1: ECCEIR[23:16] <= datain_reg;
                  8'd2: ECCEIR[15:8] <= datain_reg;
                  8'd3: begin
                    ECCEIR[7:0] <= datain_reg;
                    $display("[MODEL] ECCEIR written. ECCEIR=0x%x", 
                    {ECCEIR[31:8], datain_reg});
                  end
                endcase
              end
            endcase
          end
        end
        else begin
          if (s_datain_done)
            case(addr)
              ADDR_INTCR, 
              ADDR_ECCDIR, ADDR_ECCEIR:
                $display("[MODEL] WREN bit not set, register not written");
            endcase
        end
      end
    end

/* ECC Engine
* - In this model, no error correction applied
* - Triggered by ECC Test Enable Register and a dummy write
*/
always@(*)
if(INTCR[1]) begin // Test Enabled
  ECCDOR = ECCDIR ^ ECCEIR;

  ECCECR = 0;
  for (i=0; i<32; i=i+1) begin
    if (ECCEIR[i]) begin
      ECCECR = ECCECR + 1;
      INTCR[7] = 1;

      if(INTCR[0]) INTn_wire = 0; //Transition INTn pin
    end
  end
end

always@(INTCR) begin
  if(INTCR[6]) begin
    INTCR[7] = 0; //Reset INTRF
    INTn_wire = 1;
  end
  if(INTCR[5]) begin
    ECCECR = 0; //Reset ECC Error Count Reg
  end
end

  /* Data out control */
  always@(*) begin
    dataout_reg_latch = 0;
    dataout_reg_shift = 0;
    case(state)
      S_IDLE:
	dataout_reg = 0;

      S_CMD:
        if (s_cmd_done)
          case(datain_reg)
            RDSR: begin dataout_reg_latch = 1'b1; dataout_reg = SR; end
            RDFSR: begin dataout_reg_latch = 1'b1; dataout_reg = FSR; end
            RDID: begin dataout_reg_latch = 1'b1; dataout_reg = IDREG[31:24]; end
          endcase

      S_ADDR:
        if (s_addr_done)
          case(cmd_latch)
            READ, READ1: begin
              dataout_reg_latch = 1'b1;
              dataout_reg = mem_array[rd_addr];
            end

            RDAR: begin
              if (mlats == 4'd0) begin // Special case RDAR 0 latency
                dataout_reg_latch = 1'b1;
                case(addr)
                  ADDR_SR : dataout_reg = SR;
                  ADDR_CR1: dataout_reg = CR1;
                  ADDR_CR2: dataout_reg = CR2;
                  ADDR_FSR: dataout_reg = FSR;
                  ADDR_INTCR: dataout_reg = INTCR;
                  ADDR_ECCDIR: dataout_reg = ECCDIR[31:24];
                  ADDR_ECCEIR: dataout_reg = ECCEIR[31:24];
                  ADDR_ECCDOR: dataout_reg = ECCDOR[31:24];
                  ADDR_ECCECR: dataout_reg = ECCECR[31:24];
                  ADDR_ID : dataout_reg = IDREG[31:24];
                  default : begin 
                    dataout_reg = 0;
                    $display("[MODEL] Invalid Register Address 0x%x", addr);
                  end
                endcase
              end
            end
          endcase

      S_MODE, SD_MODE:
        if (s_mode_done && (mlats == 4'd0)) // Special case 0 latency
        begin
          case(cmd_latch)
            RDFT, DRFR, RDQO, RDQI, DRQI, RDFT1, RDQO1:
            begin
              dataout_reg_latch = 1'b1;
              dataout_reg = mem_array[rd_addr];
            end
          endcase
        end

      S_LATENCY:
        if (s_latency_done)
          case(cmd_latch)
            RDFT, DRFR, RDQO, RDQI, DRQI, RDFT1, RDQO1: 
            begin
              dataout_reg_latch = 1'b1;
              dataout_reg = mem_array[rd_addr];
            end

            RDAR: begin
              dataout_reg_latch = 1'b1;
              case(addr)
                ADDR_SR : dataout_reg = SR;
                ADDR_CR1: dataout_reg = CR1;
                ADDR_CR2: dataout_reg = CR2;
                ADDR_FSR: dataout_reg = FSR;
                ADDR_INTCR: dataout_reg = INTCR;
                ADDR_ECCDIR: dataout_reg = ECCDIR[31:24];
                ADDR_ECCEIR: dataout_reg = ECCEIR[31:24];
                ADDR_ECCDOR: dataout_reg = ECCDOR[31:24];
                ADDR_ECCECR: dataout_reg = ECCECR[31:24];
                ADDR_ID : dataout_reg = IDREG[31:24];
                default : begin 
                  dataout_reg = 0;
                  $display("[MODEL] Invalid Register Address 0x%x", addr);
                end
              endcase
            end
          endcase

      S_DATAOUT, SD_DATAOUT:
        case(cmd_latch)
          RDSR, RDFSR: 
            dataout_reg_shift = 1'b1;

          RDID: begin
            dataout_reg_latch = byte_strobe;
            dataout_reg_shift = ~byte_strobe;
            case(byte_strobe_ctr)
              8'd0: dataout_reg = IDREG[23:16];
              8'd1: dataout_reg = IDREG[15:8];
              8'd2: dataout_reg = IDREG[7:0];
            endcase
          end

          RDAR: begin
            dataout_reg_latch = byte_strobe;
            dataout_reg_shift = ~byte_strobe;
            case(addr)
              ADDR_ECCDIR:
                case(byte_strobe_ctr)
                  8'd0: dataout_reg = ECCDIR[23:16];
                  8'd1: dataout_reg = ECCDIR[15:8];
                  8'd2: dataout_reg = ECCDIR[7:0];
                endcase
              
              ADDR_ECCEIR:
                case(byte_strobe_ctr)
                  8'd0: dataout_reg = ECCEIR[23:16];
                  8'd1: dataout_reg = ECCEIR[15:8];
                  8'd2: dataout_reg = ECCEIR[7:0];
                endcase
              
              ADDR_ECCDOR:
                case(byte_strobe_ctr)
                  8'd0: dataout_reg = ECCDOR[23:16];
                  8'd1: dataout_reg = ECCDOR[15:8];
                  8'd2: dataout_reg = ECCDOR[7:0];
                endcase
              
              ADDR_ECCECR:
                case(byte_strobe_ctr)
                  8'd0: dataout_reg = ECCECR[23:16];
                  8'd1: dataout_reg = ECCECR[15:8];
                  8'd2: dataout_reg = ECCECR[7:0];
                endcase
              
              ADDR_ID:
                case(byte_strobe_ctr)
                  8'd0: dataout_reg = IDREG[23:16];
                  8'd1: dataout_reg = IDREG[15:8];
                  8'd2: dataout_reg = IDREG[7:0];
                endcase

            endcase
          end

          RDFT, DRFR, RDQO, RDQI, DRQI, RDFT1, RDQO1:
          begin
            dataout_reg_latch = byte_strobe;
            dataout_reg_shift = ~byte_strobe;
            dataout_reg = mem_array[rd_addr];
          end

          READ, READ1:
          begin
            dataout_reg_latch = byte_strobe;
            dataout_reg_shift = ~byte_strobe;
            dataout_reg = mem_array[rd_addr];
          end

        endcase

    endcase
  end

  /* Read address out of range notifier */
  reg dataout_reg_latch_d;
  always@(posedge CLK) begin
    dataout_reg_latch_d <= dataout_reg_latch;
  end
 
  /* Array write addr */
  reg [31:0] wr_ctr;
  always@(*)
    wr_addr = addr + wr_ctr;
  always@(posedge CLK)
    if ((state == S_DATAIN) || (state == SD_DATAIN)) begin
      case(cmd_latch)
        WRTE, WRFT, DRFW, WQIO, DWQO:
          if (s_datain_done)
            wr_ctr <= wr_ctr + 1;
        default:
          wr_ctr <= 0;
      endcase
    end
    else begin
      wr_ctr <= 0;
    end

  /* Array read addr */
  reg [31:0] rd_ctr;
  always@(posedge CLK)
    if ((state == S_DATAOUT) || (state == SD_DATAOUT)) begin
      case(cmd_latch)
        RDFT, RDQO, RDQI, RDFT1, RDQO1:
	 begin
          if (s_dataout_done) begin
              rd_ctr <= rd_ctr + 1;
          end
          if (rd_ctr == 1) begin
	    if(SCK_period < 100) 
		$display("[MODEL] Flagging READ. Frequency is greater than 100MHz violation");
          end
	 end
        DRFR, DRQI:
	 begin
          if (s_dataout_done) begin
              rd_ctr <= rd_ctr + 1;
          end
          if (rd_ctr == 1) begin
	    if(SCK_period < 200) 
		$display("[MODEL] Flagging READ. Frequency is greater than 50MHz violation");
          end
	 end
        READ, READ1: 
	 begin
          if (s_dataout_done) begin
              rd_ctr <= rd_ctr + 1;
          end
          if (rd_ctr == 1) begin
	    if(SCK_period < 200) 
		$display("[MODEL] Flagging READ. Frequency is greater than 50MHz violation");
          end
	 end
        default:
          rd_ctr <= 1;
      endcase
    end
    else begin // State before S_DATAOUT
        rd_ctr <= 1;
    end

  always@(*)
    case(cmd_latch)
      READ, RDFT, DRFR, RDQO, RDQI, DRQI, READ1, RDFT1, RDQO1: 
      begin
        case(state)
          S_ADDR, SD_ADDR:
            if (s_addr_done) begin 
              case(cmd_latch)
                DRFR, DRQI:
                  rd_addr <= sr_addr_ddr;
                default:
                  rd_addr <= sr_addr;
              endcase
            end
            else
              rd_addr <= 0;

          S_MODE, SD_MODE, S_LATENCY:
            rd_addr <= addr;

          S_DATAOUT, SD_DATAOUT: begin
               rd_addr <= addr + rd_ctr;
	  end
             
        endcase
      end

      default: begin
        rd_addr <= 0;
      end
    endcase

  /* Memory Array: SDR writes */
  always@(negedge CLK)
    case(cmd_latch)
      WRTE, WRFT, WQIO: begin
        if (s_datain_done)
          if (!addr_prot) begin
            mem_array[wr_addr] = datain_reg;
            $display("[MODEL] Data 0x%X written to Addr 0x%X", datain_reg, wr_addr);
          end
          else
            $display("[MODEL] Address protected or out of range: 0x%X", wr_addr);
      end
    endcase

  /* Memory Array: DDR writes */
  always@(posedge CLK or posedge CSn)
    case(cmd_latch)
      DRFW, DWQO: begin
        if (s_datain_done)
          if (!addr_prot) begin
            mem_array[wr_addr] = datain_reg;
            $display("[MODEL] Data 0x%X written to Addr 0x%X", datain_reg, wr_addr);
          end
          else
            $display("[MODEL] Address protected or out of range: 0x%X", wr_addr);
      end
    endcase

  /* Mode Latch */
  always@(posedge CLK)
    if (s_mode_done) 
      case(datain_reg[7:4]) // Latch only correct values of mode byte
        4'hA: begin
          mode_latch <= datain_reg;
          $display("[MODEL] XIP enabled");
        end
        4'hF: begin
          mode_latch <= datain_reg;
          $display("[MODEL] XIP disabled");
        end
        default: begin
          if (xip_en)
            $display("[MODEL] Invalid XIP mode byte. Retaining XIP enabled");
          else
            $display("[MODEL] Invalid XIP mode byte. Retaining XIP disabled");
        end
      endcase

  /* XIP mode enabled */
  always@(*)
    case(mode_latch[7:4])
      4'hA: xip_en = 1'b1; 
      4'hF: xip_en = 0; 
      default: xip_en = 0;
    endcase
  
  /* JEDEC Reset */
  reg [3:0] csn_ctr, si_pattern;
  wire CSn_delay;
  assign #(tCSn_del) CSn_delay = CSn;

  always@(negedge CSn_delay) begin
     si_pattern <= {si_pattern[2:0], SI_IO0}; 
  end

  always@(CLK) begin
     csn_ctr = 4'h0;
     si_pattern = 4'h0;
  end

  always@(posedge CSn_delay) begin
     if(csn_ctr < 3) begin
       if(CSn_period >= 10000)
	csn_ctr = csn_ctr + 1;
     end
     else if (csn_ctr == 3 && si_pattern == 4'h5) begin
        $display("[MODEL] Resetting device for 450us");
    	PORn = 1'b0;
    	#tRESET PORn = 1'b1;
    	mode = SPIEN;
    	for (i=0; i<ARRAY_BYTES; i=i+1) begin
      	  mem_array[i] = 0;
    	end
        $display("[MODEL] Reset done");
     end
  end
  
  /* Initializations */
  always@(posedge PORn or posedge RESETn) begin
    state = S_IDLE;
    dqout_en = 0;
    SR = SR_DEFAULT;
    CR1 = CR1_DEFAULT; 
    CR2 = CR2_DEFAULT; 
    mode_latch = 8'hF0; // XIP disabled
    INTCR = INTCR_DEFAULT;
    ECCDIR = ECCDIR_DEFAULT;
    ECCEIR = ECCEIR_DEFAULT;
    FSR = FSR_DEFAULT;
    ECCDOR = 32'h0;
    ECCECR = 32'h0;
    INTn_wire = 1;
    csn_ctr = 4'h0;
    si_pattern = 4'h0;
  end

endmodule

/*
 * Shift register data in: Serial
 */
module srdin_ser(
  input  wire       sclk,
  input  wire       cs_n,
  input  wire       din,
  output reg  [7:0] dout
  );

  always@(posedge sclk)
    if (!cs_n)
      dout <= {dout[6:0], din};

endmodule

/*
 * Shift register data in: Dual
 */
module srdin_dual(
  input  wire       sclk,
  input  wire       cs_n,
  input  wire [1:0] din,
  output reg  [7:0] dout
  );

  always@(posedge sclk)
    if (!cs_n)
      dout <= {dout[5:0], din};

endmodule

/*
 * Shift register data in: Quad
 */
module srdin_quad(
  input  wire       sclk,
  input  wire       cs_n,
  input  wire [3:0] din,
  output reg  [7:0] dout
  );

  always@(posedge sclk)
    if (!cs_n)
      dout <= {dout[3:0], din};

endmodule

/*
 * Shift register data in: Serial DDR
 */
module srdin_ser_ddr(
  input  wire       sclk,
  input  wire       cs_n,
  input  wire       din,
  output reg  [7:0] dout
  );

  always@(posedge sclk or negedge sclk)
    if (!cs_n)
      dout <= {dout[6:0], din};

endmodule

/*
 * Shift register data in: Dual DDR
 */
module srdin_dual_ddr(
  input  wire       sclk,
  input  wire       cs_n,
  input  wire [1:0] din,
  output reg  [7:0] dout
  );

  always@(posedge sclk or negedge sclk)
    if (!cs_n)
      dout <= {dout[5:0], din};

endmodule

/*
 * Shift register data in: Quad DDR
 */
module srdin_quad_ddr(
  input  wire       sclk,
  input  wire       cs_n,
  input  wire [3:0] din,
  output reg  [7:0] dout
  );

  always@(posedge sclk or negedge sclk)
    if (!cs_n)
      dout <= {dout[3:0], din};

endmodule

/*
 * Shift register data out: Serial
 */
module srdout_ser(
  input  wire       sclk,
  input  wire       cs_n,
  input  wire       rstn,
  input  wire       latch,
  input  wire       shift,
  input  wire [7:0] din,
  output wire       dout,
  output wire       ddr
  );

  reg [7:0] sr;
  assign dout = sr[7];
  reg [7:0] sr_ddr;
  assign ddr = sr_ddr[7];

  initial begin
    sr = 0;
    sr_ddr = 0;
  end

  always@(negedge sclk)
    if (!cs_n)
      case({latch, shift})
        2'b10: sr = din;
        2'b01: sr = {sr[6:0], sr[7]};
      endcase

  always@(negedge sclk or posedge sclk)
    if (!cs_n)
      case({latch, shift})
        2'b10: begin
          if (!sclk)
            sr_ddr = din; // negedge latch
          else
            sr_ddr = {sr_ddr[6:0], sr_ddr[7]}; // posedge shift
        end
        2'b01: sr_ddr = {sr_ddr[6:0], sr_ddr[7]};
      endcase

  always@(posedge rstn) begin
    sr = 0;
    sr_ddr = 0;
  end

endmodule

/*
 * Shift register data out: Dual
 */
module srdout_dual(
  input  wire       sclk,
  input  wire       cs_n,
  input  wire       rstn,
  input  wire       latch,
  input  wire       shift,
  input  wire [7:0] din,
  output wire [1:0] dout,
  output wire [1:0] ddr
  );

  reg [7:0] sr;
  assign dout = sr[7:6];
  reg [7:0] sr_ddr;
  assign ddr  = sr_ddr[7:6];

  initial begin
    sr = 0;
    sr_ddr = 0;
  end

  always@(negedge sclk)
    if (!cs_n)
      case({latch, shift})
        2'b10: sr = din;
        2'b01: sr = {sr[5:0], sr[7:6]};
      endcase

  always@(negedge sclk or posedge sclk)
    if (!cs_n)
      case({latch, shift})
        2'b10: begin
          if (!sclk)
            sr_ddr = din;
          else
            sr_ddr = {sr_ddr[5:0], sr_ddr[7:6]};
        end
        2'b01: sr_ddr = {sr_ddr[5:0], sr_ddr[7:6]};
      endcase

  always@(posedge rstn) begin
    sr = 0;
    sr_ddr = 0;
  end

endmodule

/*
 * Shift register data out: Quad
 */
module srdout_quad(
  input  wire       sclk,
  input  wire       cs_n,
  input  wire       rstn,
  input  wire       latch,
  input  wire       shift,
  input  wire [7:0] din,
  output wire [3:0] dout,
  output wire [3:0] ddr
  );

  reg [7:0] sr;
  assign dout = sr[7:4];

  reg [7:0] sr_ddr;
  assign ddr = sr_ddr[7:4];

  initial begin
    sr = 0;
    sr_ddr = 0;
  end

  always@(negedge sclk)
    if (!cs_n)
      case({latch, shift})
        2'b10: sr = din;
        2'b01: sr = {sr[3:0], sr[7:4]};
      endcase

  always@(negedge sclk or posedge sclk)
    if (!cs_n)
      case({latch, shift})
        2'b10: begin
          if (!sclk)
            sr_ddr = din;
          else
            sr_ddr = {sr_ddr[3:0], sr_ddr[7:4]};
        end
        2'b01: sr_ddr = {sr_ddr[3:0], sr_ddr[7:4]};
      endcase

  always@(posedge rstn) begin
    sr = 0;
    sr_ddr = 0;
  end

endmodule

/*
 * Address register
 */
module addr_reg(
  input  wire        sclk,
  input  wire        cs_n,
  input  wire [2:0]  mode_sel,
  input  wire [3:0]  din,
  output reg  [23:0] addr
  );

  // SPI register
  reg [23:0] spi_sr;
  always@(posedge sclk)
    if (!cs_n)
      spi_sr <= {spi_sr[22:0], din[0]};
 
  // DPI register
  reg [23:0] dpi_sr;
  always@(posedge sclk)
    if (!cs_n)
      dpi_sr <= {dpi_sr[21:0], din[1:0]};
 
  // QPI register
  reg [23:0] qpi_sr;
  always@(posedge sclk)
    if (!cs_n)
      qpi_sr <= {qpi_sr[19:0], din};
 
  always@(*)
    case(mode_sel)
      3'b001: addr <= spi_sr;
      3'b010: addr <= dpi_sr;
      3'b100: addr <= qpi_sr;
      default: addr <= 0;
    endcase

endmodule

/*
 * Address register DDR
 */
module addr_reg_ddr(
  input  wire        sclk,
  input  wire        cs_n,
  input  wire [2:0]  mode_sel,
  input  wire [3:0]  din,
  output reg  [23:0] addr
  );

  // SPI register
  reg [23:0] spi_sr;
  always@(posedge sclk or negedge sclk)
    if (!cs_n)
      spi_sr <= {spi_sr[22:0], din[0]};
 
  // DPI register
  reg [23:0] dpi_sr;
  always@(posedge sclk or negedge sclk)
    if (!cs_n)
      dpi_sr <= {dpi_sr[21:0], din[1:0]};
 
  // QPI register
  reg [23:0] qpi_sr;
  always@(posedge sclk or negedge sclk)
    if (!cs_n)
      qpi_sr <= {qpi_sr[19:0], din};
 
  always@(*)
    case(mode_sel)
      3'b001: addr <= spi_sr;
      3'b010: addr <= dpi_sr;
      3'b100: addr <= qpi_sr;
      default: addr <= 0;
    endcase

endmodule
